# 🔧 Fixing 404 Error - Shopify Routes Not Found

## 🐛 Problem
Getting 404 error when trying to access Shopify settings page.

## ✅ Solution: Register the Service Provider

The `MarketingBotServiceProvider` needs to be registered in Laravel's config.

---

## 📝 Step-by-Step Fix

### Step 1: Check if MarketingBot is Registered

Open this file on your hosting:
```
config/app.php
```

Look for this line in the `providers` array:
```php
App\Extensions\MarketingBot\System\MarketingBotServiceProvider::class,
```

**If it's NOT there**, add it!

---

### Step 2: Add to config/app.php

Find the `providers` array (around line 160) and add:

```php
'providers' => [
    // ... other providers ...
    
    /*
     * Application Service Providers...
     */
    App\Providers\AppServiceProvider::class,
    App\Providers\AuthServiceProvider::class,
    // App\Providers\BroadcastServiceProvider::class,
    App\Providers\EventServiceProvider::class,
    App\Providers\RouteServiceProvider::class,
    
    // Add this line:
    App\Extensions\MarketingBot\System\MarketingBotServiceProvider::class,
],
```

---

### Step 3: Clear Cache

Run these commands on your hosting (via SSH or Terminal in cPanel):

```bash
php artisan config:clear
php artisan route:clear
php artisan cache:clear
php artisan view:clear
```

**OR** if you don't have SSH access, delete these folders via FTP:
```
bootstrap/cache/config.php
bootstrap/cache/routes-v7.php
bootstrap/cache/services.php
```

---

### Step 4: Verify Routes

Check if routes are registered:

```bash
php artisan route:list | grep shopify
```

You should see:
```
GET|HEAD  dashboard/user/marketing-bot/shopify/settings
POST      dashboard/user/marketing-bot/shopify/connect
DELETE    dashboard/user/marketing-bot/shopify/disconnect
POST      dashboard/user/marketing-bot/shopify/sync
GET|HEAD  dashboard/user/marketing-bot/shopify/status
```

---

## 🔍 Alternative: Check if Extension is Auto-Discovered

Some Laravel apps auto-discover extensions. Check this file:
```
bootstrap/providers.php
```

Or:
```
config/extensions.php
```

The MarketingBot extension should be listed there.

---

## 🚨 Quick Fix (If Above Doesn't Work)

### Option 1: Manual Route Registration

Create/edit this file:
```
routes/web.php
```

Add at the bottom:
```php
// Shopify Integration Routes
Route::middleware(['web', 'auth'])->group(function () {
    Route::prefix('dashboard/user/marketing-bot/shopify')
        ->name('dashboard.user.marketing-bot.shopify.')
        ->group(function () {
            Route::get('settings', [App\Extensions\MarketingBot\System\Http\Controllers\ShopifyController::class, 'index'])->name('settings');
            Route::post('connect', [App\Extensions\MarketingBot\System\Http\Controllers\ShopifyController::class, 'connect'])->name('connect');
            Route::delete('disconnect', [App\Extensions\MarketingBot\System\Http\Controllers\ShopifyController::class, 'disconnect'])->name('disconnect');
            Route::post('sync', [App\Extensions\MarketingBot\System\Http\Controllers\ShopifyController::class, 'sync'])->name('sync');
            Route::get('status', [App\Extensions\MarketingBot\System\Http\Controllers\ShopifyController::class, 'syncStatus'])->name('status');
        });
    
    Route::prefix('dashboard/user/marketing-bot/dashboard')
        ->name('dashboard.user.marketing-bot.dashboard.')
        ->group(function () {
            Route::get('customize', [App\Extensions\MarketingBot\System\Http\Controllers\DashboardCustomizationController::class, 'index'])->name('customize');
            Route::post('customize', [App\Extensions\MarketingBot\System\Http\Controllers\DashboardCustomizationController::class, 'update'])->name('update');
            Route::post('reset', [App\Extensions\MarketingBot\System\Http\Controllers\DashboardCustomizationController::class, 'reset'])->name('reset');
        });
});
```

Then clear cache again:
```bash
php artisan route:clear
php artisan config:clear
```

---

## 📊 Checklist

- [ ] Check `config/app.php` has `MarketingBotServiceProvider`
- [ ] Clear all caches
- [ ] Verify routes with `php artisan route:list`
- [ ] Check file permissions (755 for folders, 644 for files)
- [ ] Verify `.htaccess` exists in public folder
- [ ] Check if mod_rewrite is enabled on hosting

---

## 🎯 Most Common Causes

1. **Service Provider not registered** (90% of cases)
2. **Cache not cleared** (5% of cases)
3. **File permissions wrong** (3% of cases)
4. **mod_rewrite disabled** (2% of cases)

---

## 💡 Quick Test

Try accessing the main Marketing Dashboard first:
```
https://yourdomain.com/dashboard/user/marketing-bot
```

**If this works**: Service provider is registered, just clear cache
**If this 404s**: Service provider is NOT registered

---

## 🆘 Still Not Working?

Send me:
1. The URL you're trying to access
2. Screenshot of the error
3. Output of `php artisan route:list | grep marketing-bot`

I'll help you debug! 😊
